// Calendário de Marketing HTM 2026 - JavaScript Principal

let currentFilter = 'all';
let isDarkMode = false;
let db; // Banco de dados IndexedDB

// Inicializa o banco de dados
function initDB() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('CalendarioMarketing', 1);

        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            db = request.result;
            resolve(db);
        };

        request.onupgradeneeded = (event) => {
            const db = event.target.result;
            if (!db.objectStoreNames.contains('anotacoes')) {
                db.createObjectStore('anotacoes', { keyPath: 'eventKey' });
            }
        };
    });
}

// Salva anotação no banco
function saveNoteToDB(eventKey, notes) {
    return new Promise((resolve, reject) => {
        const transaction = db.transaction(['anotacoes'], 'readwrite');
        const store = transaction.objectStore('anotacoes');
        const request = store.put({ eventKey: eventKey, notes: notes, timestamp: Date.now() });

        request.onsuccess = () => resolve();
        request.onerror = () => reject(request.error);
    });
}

// Busca anotação no banco
function getNoteFromDB(eventKey) {
    return new Promise((resolve, reject) => {
        const transaction = db.transaction(['anotacoes'], 'readonly');
        const store = transaction.objectStore('anotacoes');
        const request = store.get(eventKey);

        request.onsuccess = () => {
            if (request.result) {
                resolve(request.result.notes);
            } else {
                resolve('');
            }
        };
        request.onerror = () => reject(request.error);
    });
}

// Busca todas as anotações do banco
function getAllNotesFromDB() {
    return new Promise((resolve, reject) => {
        const transaction = db.transaction(['anotacoes'], 'readonly');
        const store = transaction.objectStore('anotacoes');
        const request = store.getAll();

        request.onsuccess = () => resolve(request.result);
        request.onerror = () => reject(request.error);
    });
}

// Inicializa o calendário quando a página carrega
document.addEventListener('DOMContentLoaded', async function() {
    await initDB();
    renderCalendar();
    setupEventListeners();
});

// Renderiza o calendário completo
async function renderCalendar() {
    const container = document.getElementById('calendarContainer');
    container.innerHTML = '';

    for (const monthKey of Object.keys(calendarData)) {
        const month = calendarData[monthKey];
        const monthSection = await createMonthSection(month, monthKey);
        container.appendChild(monthSection);
    }
}

// Cria a seção de um mês
async function createMonthSection(month, monthKey) {
    const section = document.createElement('div');
    section.id = monthKey;
    section.className = 'mb-12 fade-in-up';

    // Gera os cards de eventos de forma assíncrona
    const eventCards = await Promise.all(
        month.events.map(event => createEventCard(event, month))
    );

    section.innerHTML = `
        <div class="glass p-8 shadow-2xl">
            <!-- Header do Mês -->
            <div class="mb-6">
                <div class="flex items-center gap-4 mb-4">
                    <div class="w-16 h-16 bg-purple-600 flex items-center justify-center text-white text-2xl shadow-lg">
                        <i class="fas ${month.icon}"></i>
                    </div>
                    <div>
                        <h2 class="text-4xl font-bold gradient-text">${month.name} 2026</h2>
                        <p class="text-gray-600 text-lg">${month.hashtag}</p>
                    </div>
                </div>

                <div class="bg-purple-100 p-6 mb-6">
                    <h3 class="font-bold text-xl mb-2">${month.theme}</h3>
                    <p class="text-gray-700">${month.description}</p>
                </div>

                <!-- Temas em Alta -->
                <div class="mb-6">
                    <h4 class="font-bold text-lg mb-3 flex items-center gap-2">
                        <i class="fas fa-fire text-orange-500"></i> Temas em Alta
                    </h4>
                    <div class="flex flex-wrap gap-2">
                        ${month.themes.map(theme => `
                            <span class="theme-badge">
                                ${theme}
                            </span>
                        `).join('')}
                    </div>
                </div>
            </div>

            <!-- Grid de Eventos -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                ${eventCards.join('')}
            </div>
        </div>
    `;

    return section;
}

// Função auxiliar para obter o dia da semana
function getDayOfWeek(day, monthName) {
    const monthNumbers = {
        'Janeiro': 0, 'Fevereiro': 1, 'Março': 2, 'Abril': 3,
        'Maio': 4, 'Junho': 5, 'Julho': 6, 'Agosto': 7,
        'Setembro': 8, 'Outubro': 9, 'Novembro': 10, 'Dezembro': 11
    };

    const daysOfWeek = ['Domingo', 'Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta', 'Sábado'];
    const date = new Date(2026, monthNumbers[monthName], day);
    return daysOfWeek[date.getDay()];
}

// Cria um card de evento (versão assíncrona)
async function createEventCard(event, month) {
    const categoryColors = {
        feriado: 'from-red-500 to-pink-500',
        cultura: 'from-orange-500 to-yellow-500',
        data: 'from-blue-500 to-cyan-500',
        campanha: 'from-green-500 to-teal-500'
    };

    const categoryIcons = {
        feriado: 'fa-calendar-check',
        cultura: 'fa-palette',
        data: 'fa-star',
        campanha: 'fa-bullhorn'
    };

    const gradient = categoryColors[event.category] || 'from-purple-500 to-pink-500';
    const icon = categoryIcons[event.category] || 'fa-calendar-day';

    const categoryBg = {
        feriado: 'bg-red-600',
        cultura: 'bg-orange-600',
        data: 'bg-blue-600',
        campanha: 'bg-green-600'
    };

    const bg = categoryBg[event.category] || 'bg-purple-600';

    // Verifica se há anotações no banco de dados
    const eventKey = `${month.name}_${event.day}_${event.title}`;
    const notes = await getNoteFromDB(eventKey);
    const hasNotes = notes.trim() !== '';

    // Obtém o dia da semana
    const dayOfWeek = getDayOfWeek(event.day, month.name);

    return `
        <div class="day-card glass p-5 category-${event.category} event-card"
             data-category="${event.category}"
             data-title="${event.title.toLowerCase()}"
             onclick='openModal(${JSON.stringify(event)}, ${JSON.stringify(month)})'>

            <div class="flex items-start justify-between mb-3">
                <div class="flex items-center gap-3">
                    <div class="w-12 h-12 ${bg} flex items-center justify-center text-white font-bold shadow-lg">
                        ${event.day}
                    </div>
                    <div class="flex-1">
                        <div class="flex items-center gap-2 mb-1">
                            <i class="fas ${icon} text-gray-400 text-sm"></i>
                            <span class="text-xs font-semibold text-gray-500 uppercase">${event.type}</span>
                        </div>
                    </div>
                </div>
                ${hasNotes ? '<div class="text-purple-600 text-sm"><i class="fas fa-sticky-note"></i></div>' : ''}
            </div>

            <h3 class="font-bold text-lg leading-tight mb-2 text-gray-800">${event.title}</h3>
            <p class="text-sm text-gray-600 line-clamp-2">${event.description}</p>

            ${hasNotes ? `<div class="mt-3 p-3 bg-purple-50 border-l-4 border-purple-600 print-notes">
                <p class="text-xs font-semibold text-purple-800 mb-1"><i class="fas fa-edit"></i> Minhas Anotações:</p>
                <p class="text-xs text-gray-700 whitespace-pre-wrap">${notes}</p>
            </div>` : ''}

            <div class="mt-4 flex items-center justify-between">
                <span class="text-xs text-gray-400">
                    <i class="fas fa-calendar"></i> ${dayOfWeek}, ${event.day} de ${month.name}
                </span>
                <button class="text-purple-600 hover:text-purple-800 transition-colors">
                    <i class="fas fa-arrow-right"></i>
                </button>
            </div>
        </div>
    `;
}

// Abre o modal com detalhes do evento
async function openModal(event, month) {
    const modal = document.getElementById('eventModal');
    const modalTitle = document.getElementById('modalTitle');
    const modalDate = document.getElementById('modalDate');
    const modalContent = document.getElementById('modalContent');

    const categoryIcons = {
        feriado: '<i class="fas fa-calendar-check"></i>',
        cultura: '<i class="fas fa-palette"></i>',
        data: '<i class="fas fa-star"></i>',
        campanha: '<i class="fas fa-bullhorn"></i>'
    };

    const icon = categoryIcons[event.category] || '<i class="fas fa-calendar-day"></i>';
    const eventKey = `${month.name}_${event.day}_${event.title}`;
    const savedNotes = await getNoteFromDB(eventKey);
    const dayOfWeek = getDayOfWeek(event.day, month.name);

    modalTitle.innerHTML = `${icon} ${event.title}`;
    modalDate.textContent = `${dayOfWeek}, ${event.day} de ${month.name} de 2026`;

    modalContent.innerHTML = `
        <div class="space-y-6">
            <!-- Descrição -->
            <div>
                <h3 class="font-bold text-xl mb-3 flex items-center gap-2">
                    <i class="fas fa-info-circle text-purple-600"></i> Sobre este evento
                </h3>
                <p class="text-gray-700 leading-relaxed text-lg">${event.description}</p>
            </div>

            <!-- Categoria -->
            <div class="bg-purple-50 p-4">
                <div class="flex items-center gap-3">
                    <div class="w-12 h-12 bg-purple-600 flex items-center justify-center text-white">
                        <i class="fas fa-tag"></i>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Categoria</p>
                        <p class="font-bold text-lg capitalize">${event.type}</p>
                    </div>
                </div>
            </div>

            <!-- Anotações Personalizadas -->
            <div>
                <h3 class="font-bold text-xl mb-3 flex items-center gap-2">
                    <i class="fas fa-edit text-purple-600"></i> Minhas Anotações
                </h3>
                <textarea
                    id="eventNotes"
                    class="w-full p-4 border-2 border-purple-300 focus:border-purple-500 focus:outline-none transition-all min-h-32"
                    placeholder="Adicione suas ideias, planejamentos e observações sobre este evento..."
                    onchange="saveEventNotes('${eventKey}')"
                >${savedNotes}</textarea>
                <p class="text-sm text-gray-500 mt-2">
                    <i class="fas fa-info-circle"></i> Suas anotações serão salvas automaticamente e incluídas ao adicionar no Google Calendar
                </p>
            </div>

            <!-- Hashtags Sugeridas -->
            <div>
                <h3 class="font-bold text-xl mb-3 flex items-center gap-2">
                    <i class="fas fa-hashtag text-blue-500"></i> Hashtags Sugeridas
                </h3>
                <div class="flex flex-wrap gap-2">
                    ${getHashtags(event, month).map(tag => `
                        <span class="px-4 py-2 bg-blue-100 text-blue-700 font-semibold text-sm">
                            #${tag}
                        </span>
                    `).join('')}
                </div>
            </div>

            <!-- Ações -->
            <div class="flex gap-3 pt-4">
                <button onclick="addToGoogleCalendar('${event.title}', '${event.day}', '${month.name}', '${eventKey}')" class="flex-1 px-6 py-3 bg-green-600 text-white font-semibold hover:bg-green-700 transition-all">
                    <i class="fas fa-calendar-plus"></i> Adicionar ao Calendar
                </button>
                <button onclick="shareEventWhatsApp('${event.title}', '${event.day}', '${month.name}', '${eventKey}')" class="flex-1 px-6 py-3 bg-green-500 text-white font-semibold hover:bg-green-600 transition-all">
                    <i class="fab fa-whatsapp"></i> Compartilhar no WhatsApp
                </button>
            </div>
        </div>
    `;

    modal.style.display = 'block';
}

// Salva anotações do evento
async function saveEventNotes(eventKey) {
    const notes = document.getElementById('eventNotes').value;
    await saveNoteToDB(eventKey, notes);

    // Re-renderiza o calendário para atualizar o indicador visual
    await renderCalendar();

    // Mantém o filtro atual ativo
    if (currentFilter !== 'all') {
        setTimeout(() => {
            const button = document.querySelector(`.sidebar-btn[onclick*="'${currentFilter}'"]`);
            if (button) {
                button.click();
            }
        }, 100);
    }
}

// Fecha o modal
function closeModal() {
    const modal = document.getElementById('eventModal');
    modal.style.display = 'none';
}

// Fecha modal ao clicar fora dele
window.onclick = function(event) {
    const modal = document.getElementById('eventModal');
    if (event.target === modal) {
        closeModal();
    }
}

// Gera hashtags sugeridas
function getHashtags(event, month) {
    const baseHashtags = [
        event.title.replace(/\s+/g, ''),
        month.name + '2026',
        'Marketing2026'
    ];

    const categoryHashtags = {
        feriado: ['Feriado', 'Celebração', 'Promoção'],
        cultura: ['Cultura', 'Arte', 'Brasil'],
        data: ['DataComemorativa', 'Celebrar'],
        campanha: ['Campanha', 'Conscientização', 'Sociedade']
    };

    return [...baseHashtags, ...(categoryHashtags[event.category] || [])].slice(0, 6);
}

// Filtra eventos por categoria
function filterCategory(category) {
    currentFilter = category;

    // Atualiza visual dos botões de filtro da sidebar
    document.querySelectorAll('.sidebar-btn[onclick*="filterCategory"]').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');

    // Filtra os cards
    const cards = document.querySelectorAll('.event-card');
    cards.forEach(card => {
        if (category === 'all' || card.dataset.category === category) {
            card.style.display = 'block';
            card.classList.add('fade-in-up');
        } else {
            card.style.display = 'none';
        }
    });

    // Fecha sidebar em mobile após selecionar
    if (window.innerWidth <= 1024) {
        toggleSidebar();
    }
}

// Toggle sidebar (mobile)
function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    const overlay = document.querySelector('.sidebar-overlay');

    sidebar.classList.toggle('open');
    overlay.classList.toggle('open');
}

// Scroll suave para o topo
function scrollToTop() {
    window.scrollTo({
        top: 0,
        behavior: 'smooth'
    });
}

// Mostra/esconde botão "Voltar ao topo"
function toggleBackToTopButton() {
    const backToTopBtn = document.getElementById('backToTop');
    if (window.pageYOffset > 300) {
        backToTopBtn.classList.add('show');
    } else {
        backToTopBtn.classList.remove('show');
    }
}

// Event listener para scroll
window.addEventListener('scroll', toggleBackToTopButton);

// Busca eventos
function searchEvents() {
    const searchTerm = document.getElementById('searchInput').value.toLowerCase();
    const cards = document.querySelectorAll('.event-card');

    cards.forEach(card => {
        const title = card.dataset.title;
        if (title.includes(searchTerm)) {
            card.style.display = 'block';
        } else {
            card.style.display = 'none';
        }
    });
}

// Rola até o mês específico
function scrollToMonth(monthKey) {
    const monthElement = document.getElementById(monthKey);
    if (monthElement) {
        monthElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }

    // Fecha sidebar em mobile após selecionar
    if (window.innerWidth <= 1024) {
        toggleSidebar();
    }
}

// Alterna tema claro/escuro
function toggleTheme() {
    isDarkMode = !isDarkMode;
    document.body.classList.toggle('dark-mode');

    const glassElements = document.querySelectorAll('.glass');
    glassElements.forEach(el => {
        if (isDarkMode) {
            el.classList.add('glass-dark');
            el.classList.remove('glass');
        } else {
            el.classList.add('glass');
            el.classList.remove('glass-dark');
        }
    });
}

// Exporta o calendário
function exportCalendar() {
    const options = `
        <div class="text-center">
            <h3 class="text-2xl font-bold mb-6">Exportar Calendário</h3>
            <div class="grid grid-cols-1 gap-4">
                <button onclick="exportToPDF()" class="px-6 py-4 bg-red-600 text-white font-semibold hover:bg-red-700 transition-all">
                    <i class="fas fa-file-pdf text-2xl mb-2"></i><br>
                    Exportar como PDF
                </button>
                <button onclick="exportToExcel()" class="px-6 py-4 bg-green-600 text-white font-semibold hover:bg-green-700 transition-all">
                    <i class="fas fa-file-excel text-2xl mb-2"></i><br>
                    Exportar como Excel
                </button>
                <button onclick="exportToGoogleCalendar()" class="px-6 py-4 bg-blue-600 text-white font-semibold hover:bg-blue-700 transition-all">
                    <i class="fab fa-google text-2xl mb-2"></i><br>
                    Exportar para Google Calendar
                </button>
            </div>
        </div>
    `;

    document.getElementById('modalTitle').innerHTML = '<i class="fas fa-download"></i> Exportar Calendário';
    document.getElementById('modalDate').textContent = 'Escolha o formato desejado';
    document.getElementById('modalContent').innerHTML = options;
    document.getElementById('eventModal').style.display = 'block';
}

// Exporta para PDF
function exportToPDF() {
    // Usa o print do navegador como solução para PDF
    closeModal();
    setTimeout(() => {
        window.print();
    }, 300);
}

// Exporta para Excel
async function exportToExcel() {
    let csvContent = "data:text/csv;charset=utf-8,";
    csvContent += "Mês,Dia,Título,Tipo,Categoria,Descrição,Minhas Anotações\n";

    for (const monthKey of Object.keys(calendarData)) {
        const month = calendarData[monthKey];
        for (const event of month.events) {
            const eventKey = `${month.name}_${event.day}_${event.title}`;
            const notes = await getNoteFromDB(eventKey);
            const notesEscaped = notes.replace(/"/g, '""'); // Escape aspas duplas no CSV

            csvContent += `${month.name},${event.day},"${event.title}",${event.type},${event.category},"${event.description}","${notesEscaped}"\n`;
        }
    }

    const encodedUri = encodeURI(csvContent);
    const link = document.createElement("a");
    link.setAttribute("href", encodedUri);
    link.setAttribute("download", "calendario_marketing_2026.csv");
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);

    showSuccessMessage('Arquivo CSV exportado com suas anotações!');
    closeModal();
}

// Exporta para Google Calendar
async function exportToGoogleCalendar() {
    closeModal();

    // Cria arquivo ICS (iCalendar) para importar no Google Calendar
    let icsContent = 'BEGIN:VCALENDAR\n';
    icsContent += 'VERSION:2.0\n';
    icsContent += 'PRODID:-//Calendário de Marketing 2026//PT\n';
    icsContent += 'CALSCALE:GREGORIAN\n';
    icsContent += 'METHOD:PUBLISH\n';
    icsContent += 'X-WR-CALNAME:Calendário de Marketing 2026\n';
    icsContent += 'X-WR-TIMEZONE:America/Sao_Paulo\n';

    const monthNumbers = {
        'Janeiro': '01', 'Fevereiro': '02', 'Março': '03', 'Abril': '04',
        'Maio': '05', 'Junho': '06', 'Julho': '07', 'Agosto': '08',
        'Setembro': '09', 'Outubro': '10', 'Novembro': '11', 'Dezembro': '12'
    };

    for (const monthKey of Object.keys(calendarData)) {
        const month = calendarData[monthKey];
        for (const event of month.events) {
            const date = `2026${monthNumbers[month.name]}${event.day.toString().padStart(2, '0')}`;
            const eventKey = `${month.name}_${event.day}_${event.title}`;
            const notes = await getNoteFromDB(eventKey);

            // Monta descrição completa
            let description = event.description;
            if (notes) {
                description += `\\n\\n--- MINHAS ANOTAÇÕES ---\\n${notes.replace(/\n/g, '\\n')}`;
            }

            icsContent += 'BEGIN:VEVENT\n';
            icsContent += `DTSTART;VALUE=DATE:${date}\n`;
            icsContent += `DTEND;VALUE=DATE:${date}\n`;
            icsContent += `SUMMARY:${event.title}\n`;
            icsContent += `DESCRIPTION:${description}\n`;
            icsContent += `CATEGORIES:${event.category}\n`;
            icsContent += `UID:${monthKey}-${event.day}-${event.title.replace(/\s+/g, '-')}@marketing2026\n`;
            icsContent += 'END:VEVENT\n';
        }
    }

    icsContent += 'END:VCALENDAR';

    // Download do arquivo ICS
    const blob = new Blob([icsContent], { type: 'text/calendar;charset=utf-8' });
    const link = document.createElement('a');
    link.href = window.URL.createObjectURL(blob);
    link.setAttribute('download', 'calendario_marketing_2026.ics');
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);

    showSuccessMessage('Arquivo ICS exportado com suas anotações! Importe no Google Calendar.');
}

// Adiciona evento ao Google Calendar
async function addToGoogleCalendar(title, day, month, eventKey) {
    const monthNumbers = {
        'Janeiro': '01', 'Fevereiro': '02', 'Março': '03', 'Abril': '04',
        'Maio': '05', 'Junho': '06', 'Julho': '07', 'Agosto': '08',
        'Setembro': '09', 'Outubro': '10', 'Novembro': '11', 'Dezembro': '12'
    };

    const date = `2026${monthNumbers[month]}${day.toString().padStart(2, '0')}`;

    // Inclui as anotações personalizadas do banco
    const notes = await getNoteFromDB(eventKey);
    let details = 'Evento do Calendário de Marketing 2026';
    if (notes) {
        details += '\n\nMinhas Anotações:\n' + notes;
    }

    const url = `https://calendar.google.com/calendar/render?action=TEMPLATE&text=${encodeURIComponent(title)}&dates=${date}/${date}&details=${encodeURIComponent(details)}&sf=true&output=xml`;

    window.open(url, '_blank');
}

// Compartilha evento no WhatsApp
async function shareEventWhatsApp(title, day, month, eventKey) {
    const notes = await getNoteFromDB(eventKey);

    let text = `*${title}*\n`;
    text += `${day} de ${month} de 2026\n`;

    if (notes) {
        text += `\n*Minhas Anotações:*\n${notes}\n`;
    }

    text += `\n#CalendárioMarketing #Marketing2026`;

    // Codifica o texto para URL
    const encodedText = encodeURIComponent(text);

    // Abre o WhatsApp com a mensagem pré-formatada
    const whatsappUrl = `https://wa.me/?text=${encodedText}`;

    window.open(whatsappUrl, '_blank');
}

// Copia texto para área de transferência
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(() => {
        showSuccessMessage('Texto copiado para a área de transferência!');
    }).catch(err => {
        // Fallback para navegadores antigos
        const textarea = document.createElement('textarea');
        textarea.value = text;
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        showSuccessMessage('Texto copiado para a área de transferência!');
    });
}

// Mostra mensagem de sucesso
function showSuccessMessage(message) {
    const notification = document.createElement('div');
    notification.className = 'fixed top-4 right-4 bg-green-600 text-white px-6 py-4 shadow-2xl z-50 animate-fade-in';
    notification.innerHTML = `
        <div class="flex items-center gap-3">
            <i class="fas fa-check-circle text-2xl"></i>
            <span class="font-semibold">${message}</span>
        </div>
    `;

    document.body.appendChild(notification);

    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateY(-20px)';
        notification.style.transition = 'all 0.3s ease';
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}

// Compartilha calendário completo
function shareCalendar() {
    const text = `Calendário de Marketing 2026 🚀\n\nDatas, tendências e oportunidades para 2026!\n\n✅ 270+ Datas Comemorativas\n✅ 50+ Eventos Culturais\n✅ 12 Campanhas Sazonais\n\nAcesse: ${window.location.href}\n\n#CalendárioMarketing #Marketing2026`;

    if (navigator.share) {
        navigator.share({
            title: 'Calendário de Marketing 2026',
            text: text,
            url: window.location.href
        }).then(() => {
            showSuccessMessage('Calendário compartilhado com sucesso!');
        }).catch(err => {
            if (err.name !== 'AbortError') {
                copyToClipboard(text);
            }
        });
    } else {
        copyToClipboard(text);
    }
}

// Compartilha mês completo
function shareMonth(monthKey) {
    const month = calendarData[monthKey];
    const text = `Calendário de Marketing - ${month.name} 2026\n${month.hashtag}\n\n${month.description}\n\n#CalendárioMarketing #Marketing2026`;

    if (navigator.share) {
        navigator.share({
            title: `${month.name} 2026 - Calendário de Marketing`,
            text: text,
            url: window.location.href + '#' + monthKey
        }).then(() => {
            showSuccessMessage('Mês compartilhado com sucesso!');
        }).catch(err => {
            if (err.name !== 'AbortError') {
                copyToClipboard(text);
            }
        });
    } else {
        copyToClipboard(text);
    }
}

// Configura event listeners
function setupEventListeners() {
    // Esc para fechar modal
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeModal();
        }
    });
}

// Animação de entrada dos elementos
const observerOptions = {
    threshold: 0.1,
    rootMargin: '0px 0px -100px 0px'
};

const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
        if (entry.isIntersecting) {
            entry.target.style.opacity = '1';
            entry.target.style.transform = 'translateY(0)';
        }
    });
}, observerOptions);

// Observa elementos para animação
document.addEventListener('DOMContentLoaded', () => {
    const elements = document.querySelectorAll('.fade-in-up');
    elements.forEach(el => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(30px)';
        el.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(el);
    });
});
